﻿/*	VERSION: 1.0


USAGE: 
	// This must be included in each scope where clipboard key-combos are trying to be detected.  (before your key detection code is used)
	// if omitted later on,  the events will fire,  but the keyCodes will be wrong.
	#include "functions/zincClipboardKeys.as"
	
	Key.addListener( {
		onKeyDown: function(){
			mdm.Exception.DebugWindow.trace("onKeyDown  key: " + Key.getCode() );
		}
	} );
	
	
QUIRKS: 
	"onKeyUp" is unreliable,  because it does not occur upon releasing the 2nd key of a key-combo.  Depending on which order you release the keys, that might be the letter or the CTRL key.
	
	
DESCRIPTION:
	MDM Zinc normally blocks certain common clipboard combinations.  Specifically these: 
	a z x c v
	This code allows normal "onKeyDown" and "onKeyUp" events to occur for those combinations,  so they can be detected just like any other key or combination.  ("onKeyUp" is currently unreliable)
	
	
HOW??
	This creates a local "var Key" object which functions exactly like the real "_global.Key" class,  but it contains a modified Key.getCode() function.
	The actual code exists globally and only actually instantiates once  (if Zinc exists).  It continuously polls for blocked key combinations and manually fires the "onKeyDown" and "onKeyUp" events.
*/
// one-time init
function runFunc( func ){
	return func.apply( _this, arguments.slice() );
}// runFunc()
if( _global.custom_getCode === undefined  &&  _global.mdm ){
	runFunc(function(){
		_global.custom_getCode = Key.getCode();
		var lastKeyCode = _global.Key.getCode();
		// Which key conbinations does Zinc block?		a z x c v
		var a_code = String("A").toUpperCase().charCodeAt(0);		// get the ascii for the letter "a"		(uppercase ascii letters/numbers === letter/number key-codes)
		var z_code = String("Z").toUpperCase().charCodeAt(0);
		var x_code = String("X").toUpperCase().charCodeAt(0);
		var c_code = String("C").toUpperCase().charCodeAt(0);
		var v_code = String("V").toUpperCase().charCodeAt(0);
		
		
		var reactions = {
			onKeyDown: function(){
				// clean-up
				if(_global.custom_getCode === undefined){
					cleanUp();
					break;
				}// if:  custom_getCode was removed
			}, // onKeyDown()
			
			
			onKeyUp: function(){
				// detect when Key.getCode() actually changes
				if( _global.Key.getCode() !== lastKeyCode ){
					_global.custom_getCode = 0;		// null does work,  but seems risky since the clean-up code looks for "undefined"
					lastKeyCode = _global.Key.getCode();
				}// if:  key-code changed
				lastCodeSent = null;
			}// onKeyUp()
		}// reactions {}
		Key.addListener( reactions );
		
		
		var lastCodeSent = null;
		var loop = setInterval(function(){
			// clean-up
			if(_global.my_key_class === undefined){
				cleanUp();
				break;
			}// if:  custom_getCode was removed
			// detect blocked CTRL+?  key-combinations...		otherwise abort
			if( Key.isDown( Key.CONTROL ) === false )		return;
			// CTRL is being held
			// check for CTRL+? key-combinations
			var codeSent = null;
			if(!codeSent)		codeSent = checkForKey( a_code );
			if(!codeSent)		codeSent = checkForKey( z_code );
			if(!codeSent)		codeSent = checkForKey( x_code );
			if(!codeSent)		codeSent = checkForKey( c_code );
			if(!codeSent)		codeSent = checkForKey( v_code );
			// don't repeat while being held
			lastCodeSent = codeSent;
			// no combos held  =>  re-enable sending codes  (equivalent to onKeyUp condition)
			if( !codeSent ){
				send( "onKeyUp", lastCodeSent );
				lastCodeSent = null;
			}
		}, 33);
		
		
		function checkForKey( keyCode ){
			var output = null;
			if( Key.isDown( keyCode ) ){
				output = keyCode;
				if( lastCodeSent !== keyCode ){
					send( "onKeyDown", keyCode );
				}// if:  not already sending this key
			}// if:  this key is being held
			return output;
		}// checkForKey()
		
		
		function send( eventName, keyCode ){
			if( !keyCode )		return;
			_global.my_key_class.setCode( keyCode );
			_global.my_key_class["broadcastMessage"]( eventName );
		}// send()
		
		
		function cleanUp(){
			delete _global.custom_getCode;
			clearInterval( loop );
			Key.removeListener( reactions );
		}// cleanUp()
		
		
		_global.my_key_class = {
			_listeners:null,
			ALT:null,
			BACKSPACE:null,
			CAPSLOCK:null,
			CONTROL:null,
			INSERT:null,
			DELETEKEY:null,
			DOWN:null,
			END:null,
			ENTER:null,
			ESCAPE:null,
			HOME:null,
			LEFT:null,
			PGDN:null,
			PGUP:null,
			RIGHT:null,
			SHIFT:null,
			SPACE:null,
			TAB:null,
			UP:null,
			
			addListener:null,
			getAscii:null,
			getCode:null,
			isDown:null,
			isToggled:null,
			removeListener:null,
			isAccessible:null, 
			
			broadcastMessage: null
		}
		// my_key_class can be used in-place of Key
		for(var nam in my_key_class){
			my_key_class[ nam ] = _global.Key[ nam ];
		}
		// I CAN over-write getCode() in my own "copy" of Key
		my_key_class.getCode = function(){
			return _global.custom_getCode || _global.Key.getCode();
		}// getCode()
		my_key_class.setCode = function( newValue ){
			_global.custom_getCode = newValue;
		}// setCode()

		
	});// runFunc()
}// first time init... if Zinc exists


// "Key" cannot be globally replaced
// _global.Key = _global.my_key_class;		// doesn't work
if( _global.my_key_class )		var Key = _global.my_key_class;						// this works